<?php

namespace Paheko\Accounting;

use Paheko\Entities\Accounting\Account;
use Paheko\Entities\Accounting\Chart;
use Paheko\Utils;
use Paheko\DB;
use Paheko\UserException;

use KD2\DB\EntityManager;

use const Paheko\ROOT;

class Charts
{
	const BUNDLED_CHARTS = [
		'fr_pca_2025' => 'Plan comptable associatif (révision 2025)',
		'fr_pcc_2020' => 'Plan comptable des copropriétés (2005 révisé en 2024)',
		'fr_cse_2015' => 'Plan comptable des CSE (Comité Social et Économique) (Règlement ANC n°2015-01)',
		'fr_pcg_2025' => 'Plan comptable général, pour entreprises (révision 2025)',
		'fr_pcs_2018' => 'Plan comptable des syndicats (2018, révisé 2024)',
		'be_pcmn_2019' => 'Plan comptable minimum normalisé des associations et fondations 2019',
		'ch_asso' => 'Plan comptable associatif',
		'fr_pca_1999' => 'Plan comptable associatif 1999',
		'fr_pca_2018' => 'Plan comptable associatif (2018, révision 2024)',
		'fr_pcg_2014' => 'Plan comptable général, pour entreprises (2014, révision 2024)',
	];

	const COUNTRIES_CHARTS = [
		'FR' => [
			'fr_pca_2025' => 'Association ou fondation',
			'fr_cse_2015' => 'CSE (Comité Social et Économique)',
			'fr_pcs_2018' => 'Syndicat',
			'fr_pcc_2020' => 'Copropriété',
			'fr_pcg_2025' => 'Entreprise',
		],
		'CH' => [
			'ch_asso' => 'Association',
		],
		'BE' => [
			'be_pcmn_2019' => 'Association ou fondation',
		],
	];

	static public function updateInstalled(string $chart_code): ?Chart
	{
		if (!isset(self::BUNDLED_CHARTS[$chart_code])) {
			throw new \LogicException('This chart code does not exist: ' . $chart_code);
		}

		$file = sprintf('%s/include/data/charts/%s.csv', ROOT, $chart_code);
		$country = strtoupper(substr($chart_code, 0, 2));
		$code = strtoupper(substr($chart_code, 3));

		$chart = EntityManager::findOne(Chart::class, 'SELECT * FROM @TABLE WHERE code = ? AND country = ?;', $code, $country);

		if (!$chart) {
			return null;
		}

		$chart->set('label', self::BUNDLED_CHARTS[$chart_code]);
		$chart->save();

		$chart->importCSV($file, true);
		return $chart;
	}

	static public function resetRules(array $country_list): void
	{
		foreach (self::list() as $c) {
			if (in_array($c->country, $country_list)) {
				$c->resetAccountsRules();
			}
		}
	}

	static public function installCountryDefault(string $country_code): Chart
	{
		if ($country_code == 'CH') {
			$chart_code = 'ch_asso';
		}
		elseif ($country_code == 'BE') {
			$chart_code = 'be_pcmn_2019';
		}
		else {
			$chart_code = 'fr_pca_2025';
		}

		return self::install($chart_code);
	}

	static public function install(string $chart_code): Chart
	{
		if (!array_key_exists($chart_code, self::BUNDLED_CHARTS)) {
			throw new \InvalidArgumentException('Le plan comptable demandé n\'existe pas.');
		}

		$file = sprintf('%s/include/data/charts/%s.csv', ROOT, $chart_code);

		if (!file_exists($file)) {
			throw new \LogicException('Le plan comptable demandé n\'a pas de fichier CSV');
		}

		$country = strtoupper(substr($chart_code, 0, 2));
		$code = strtoupper(substr($chart_code, 3));

		if (DB::getInstance()->test(Chart::TABLE, 'country = ? AND code = ?', $country, $code)) {
			throw new UserException('Ce plan comptable est déjà installé');
		}

		$db = DB::getInstance();
		$db->begin();

		$chart = new Chart;

        $chart->label = self::BUNDLED_CHARTS[$chart_code];
        $chart->country = $country;
        $chart->code = $code;
        $chart->save();
        $chart->importCSV($file);

        $db->commit();
        return $chart;
	}

	static public function listInstallable(): array
	{
		$installed = DB::getInstance()->getAssoc('SELECT id, LOWER(country || \'_\' || code) FROM acc_charts;');
		$out = [];

		foreach (self::BUNDLED_CHARTS as $code => $label) {
			if (in_array($code, $installed)) {
				continue;
			}

			$out[$code] = sprintf('%s — %s', Utils::getCountryName(substr($code, 0, 2)), $label);
		}

		return $out;
	}

	static public function get(int $id): ?Chart
	{
		return EntityManager::findOneById(Chart::class, $id);
	}

	static public function getByCode(string $chart_code): ?Chart
	{
		$country = strtoupper(substr($chart_code, 0, 2));
		$code = strtoupper(substr($chart_code, 3));
		return EntityManager::findOne(Chart::class, 'SELECT * FROM @TABLE WHERE code = ? AND country = ?;', $code, $country);
	}

	static public function list(): array
	{
		$em = EntityManager::getInstance(Chart::class);
		return $em->all('SELECT * FROM @TABLE ORDER BY country, label;');
	}

	static public function count(): int
	{
		return EntityManager::getInstance(Chart::class)->count();
	}

	static public function listByCountryFirstSetup(): array
	{
		$db = DB::getInstance();

		$out = self::COUNTRIES_CHARTS;

		$sql = sprintf('SELECT id, country, code, label FROM %s ORDER BY label COLLATE U_NOCASE;', Chart::TABLE);

		foreach ($db->iterate($sql) as $chart) {
			$country = strtoupper($chart->country ?? '') ?: 'other';

			if (!isset($out[$country])) {
				$out[$country] = [];
			}

			$key = $chart->code ? strtolower($chart->country . '_' . $chart->code) : $chart->id;

			$out[$country][$key] = $chart->label;
		}

		return $out;
	}

	static public function hasCustomCharts(): bool
	{
		return DB::getInstance()->test(Chart::TABLE, 'country IS NULL OR code IS NULL');
	}

	static public function getOrInstall(string $id_or_code): int
	{
		if (ctype_digit($id_or_code)) {
			return (int) $id_or_code;
		}

		$country = strtoupper(substr($id_or_code, 0, 2));
		$code = strtoupper(substr($id_or_code, 3));
		$id = DB::getInstance()->firstColumn('SELECT id FROM acc_charts WHERE country = ? AND code = ?;', $country, $code);

		if ($id) {
			return $id;
		}

		$chart = self::install($id_or_code);
		return $chart->id;
	}

	static public function listByCountry(bool $filter_archived = false)
	{
		$where = $filter_archived ? ' AND archived = 0' : '';
		$sql = sprintf('SELECT id, country, label FROM %s WHERE 1 %s ORDER BY country, code DESC, label;', Chart::TABLE, $where);
		$list = DB::getInstance()->getGrouped($sql);
		$out = [];

		foreach ($list as $row) {
			$country = $row->country ? Utils::getCountryName($row->country) : 'Aucun';

			if (!array_key_exists($country, $out)) {
				$out[$country] = [];
			}

			$out[$country][$row->id] = $row->label;
		}

		return $out;
	}

	static public function copyFrom(int $from_id, ?string $label, ?string $country): void
	{
		$db = DB::getInstance();
		$db->begin();

		$chart = new Chart;
		$chart->importForm(compact('label', 'country'));
		$chart->save();

		$db->exec(sprintf('INSERT INTO %s (id_chart, code, label, description, position, type, user, bookmark)
			SELECT %d, code, label, description, position, type, user, bookmark FROM %1$s WHERE id_chart = %d;', Account::TABLE, $chart->id, $from_id));
		$db->commit();
	}

	static public function import(string $file_key, ?string $label, ?string $country): void
	{
		if (empty($_FILES[$file_key]) || empty($_FILES[$file_key]['size']) || empty($_FILES[$file_key]['tmp_name'])) {
			throw new UserException('Fichier invalide');
		}

		$db = DB::getInstance();
		$db->begin();

		$chart = new Chart;
		$chart->importForm(compact('label', 'country'));
		$chart->save();
		$chart->importCSV($_FILES[$file_key]['tmp_name']); // This will save everything

		$db->commit();
	}

	/**
	 * Migrate old accounting charts to 2025
	 * @todo FIXME: Put this in Upgrade migration for 1.3.15 version
	 */
	static public function migrateTo2025(): void
	{
		$db = DB::getInstance();

		if (!self::getByCode('fr_pca_2025')
			&& ($old_chart = self::getByCode('fr_pca_2018'))) {
			$new_chart = self::install('fr_pca_2025');

			// Copy old accounts
			$db->exec(sprintf('INSERT OR IGNORE INTO acc_accounts (id_chart, code, label, description, position, type, user, bookmark)
				SELECT %d, code, label, description, position, type, user, bookmark
				FROM acc_accounts
				WHERE id_chart = %d AND user = 1;', $new_chart->id(), $old_chart->id()));
		}

		if (!self::getByCode('fr_pcg_2025')
			&& ($old_chart = self::getByCode('fr_pcg_2014'))) {
			$new_chart = self::install('fr_pcg_2025');

			// Copy old accounts
			$db->exec(sprintf('INSERT OR IGNORE INTO acc_accounts (id_chart, code, label, description, position, type, user, bookmark)
				SELECT %d, code, label, description, position, type, user, bookmark
				FROM acc_accounts
				WHERE id_chart = %d AND user = 1;', $new_chart->id(), $old_chart->id()));
		}
	}
}
